/*global define */
/*jslint white: true, nomen: true */


/*
	treeContainer:

	This object provides a mixin for each tree container item.
*/

define(function () {
	"use strict";

	var arrayPush = Array.prototype.push;

	function identity(obj) {
		return obj;
	}

	function mixin(options0) {
		var options = options0 || {},
			childrenL_ = options.childrenLabel || "children_",
			treeInfo_ = options.getTreeInfo || identity;

		return {

			getChildren: function () {
				return treeInfo_(this)[childrenL_];
			},

			addChild: function (child, index0) {
				var info = treeInfo_(this);
				this.removeChild(child);
				if (index0 === undefined) {
					info[childrenL_].push(child);
				} else {
					info[childrenL_].splice(index0, 0, child);
				}
				child.setParent(this, index0);
			},

			getIndexOfChild: function (child) {
				var info = treeInfo_(this), index = info[childrenL_].indexOf(child);
				return index;
			},

			getChildFromIndex: function (idx) {
				return this.getChildren()[idx];
			},

			removeChild: function (child) {
				var index = this.getIndexOfChild(child);

				if (index === -1) { return null; }

				return this.removeChildAtIndex(index);
			},

			removeChildAtIndex: function (index) {
				var info = treeInfo_(this),
					removed = info[childrenL_].splice(index, 1);

				if (removed.length > 0) {
					removed[0].setParent(null);
					// Handle.js used this semantics...
					// return removed[0];
					// ...but Container.js expects this semantics.
					// TODO: resolve
					return true;
				}

				return false;
			},

			removeChildren: function (numToRemove) {
				var c;

				if (numToRemove) {
					numToRemove = Math.min(numToRemove, this.m_children.length);
				} else {
					numToRemove = this.m_children.length;
				}

				for (c = numToRemove - 1; c >= 0; c -= 1) {
					this.removeChildAtIndex(c);
				}
			},


			findNodeRegExp: function (inRegExps, isAbsolute) {
				var head;

				function traverseNodesNamed(head, regexps) {
					var i, j, found, children;
					for (i = 0; i < regexps.length; i += 1) {
						found = null;
						children = head.getChildren && head.getChildren();
						if (children) {
							for (j = 0; j < children.length; j += 1) {
								if (regexps[i].test(children[j].getName())) {
									found = children[j];
									break;
								}
							}
						}

						if (found) {
							head = found;
						} else {
							return null;
						}
					}

					return head;
				}

				function findNodeNamed(head, regexp) { // searches deeply through entire tree, including root
					var j, found, children;

					if (regexp.test(head.getName())) {
						return head;
					}

					children = head.getChildren && head.getChildren();
					if (children) {
						for (j = 0; j < children.length; j += 1) {
							found = findNodeNamed(children[j], regexp);
							if (found) {
								return found;
							}
						}
					}

					return null;
				}

				if (isAbsolute) {
					if (inRegExps[0].test(this.getName())) {
						return traverseNodesNamed(this, inRegExps.slice(1));
					}

					return null;
				}

				// else 

				// arbitrary path
				head = findNodeNamed(this, inRegExps[0]);

				if (head) {
					return traverseNodesNamed(head, inRegExps.slice(1));
				}

				return null;
			},

			// inPath is a path, either absolute ("/a/b/c") or relative ("b/c")
			findNodePath: function (inPath) {
				var names = inPath.split("/"), isAbsolute = (inPath.charAt(0) === "/"), regExpArray = [], n;

				function stringToRegExp(s) {
					return s.replace(/[\-\/\\\^$*+?.()|\[\]{}]/g, "\\$&");
				}

				for (n = 0; n < names.length; n += 1) {
					if (names[n].length > 0) {
						regExpArray.push(new RegExp("^" + stringToRegExp(names[n]) + "$"));
					}
				}

				return this.findNodeRegExp(regExpArray, isAbsolute);
			},

			findNode: function (inPath) { // for test code only -- all behaviors should use layer params instead
				return this.findNodePath(inPath);
			},

			// return array of nodes that match regexp
			findAll: function (regexp) {
				function findAllInSubtree(subtree) {
					var j, partial, total = [],
						children = subtree.getChildren && subtree.getChildren();

					if (children) {
						for (j = 0; j < children.length; j += 1) {
							partial = findAllInSubtree(children[j]);
							if (partial) {
								total = total.concat(partial);
							}
						}
					}

					if (regexp.test(subtree.getName())) {
						total.push(subtree);
					}

					return total;
				}

				// arbitrary path
				return findAllInSubtree(this);
			},

			// bread-first traversal
			// callback functions may break early by explicitly returning false
			breadthFirstEach: function (fn) {
				var i, node, children,
					queue = [this];

				while (queue.length > 0) {
					node = queue.shift();
					if (fn(node) === false) { break; }
					children = node.getChildren && node.getChildren();

					if (children) {
						for (i = 0; i < children.length; i += 1) {
							queue.push(children[i]);
						}
					}
				}

				return this;
			},

			/**
			* Invoke a function on each tree item in a pre-order traversal.
			* Unless the function ends traversal by returning false value.
			* @param fn Function to apply on each node.
			*/
			preOrderEach: function (fn) {
				var arg = this,
					stack = [];

				while (arg) {
					if (fn(arg) === false) { break; }
					var kids = arg.getChildren && arg.getChildren().slice(0);
					arg = kids.shift();
					if (arg) {
						arrayPush.apply(stack, kids.reverse());
					} else {
						arg = stack.pop();
					}
				}

				return this;
			}
		};
	}

	return mixin;
});
